##################### tf-idf mit dem Paket Tidytext ############################

# Zunächst und für alle Fälle: Arbeitsbereich komplett bereinigen (Housekeeping)
rm(list= ls(all= TRUE))

# Ggf. notwendige Pakete installieren; die müssten Sie eigentlich alle haben...
# install.packages("tidytext")
# install.packages("dplyr")
# install.packages("ggplot2")
# install.packages("quanteda")
# install.packages("readtext")

# Aufrufen dieser Pakete
library(tidytext)
library(dplyr)
library(ggplot2)
library(quanteda)
library(readtext)

# Warum heißt das Paket eigentlich "tidytext"? 
# Hadley Wickham hat das Konzept der "tidy data" entwickelt, siehe
# https://doi.org/10.18637/jss.v059.i10

# Grundidee: "tidy data" haben eine spezifische Struktur:
# Jede Variable bildet eine eigene Spalte
# Jede Beobachtung bildet eine eigene Zeile
# Jeder Typ einer beobachtbaren Einheit bildet eine Tabelle
# Aufbauend auf den hier genannten Prinzipen hat Julia Silge,
# Autorin des Pakets "tidytext", das Format
# einer Table mit einem token pro Zeile ("one token per row")
# implementiert.

# Das sieht dann folgendermaßen aus:
text <- c("Der Flügelflagel gaustert",
          "durchs Wiruwaruwolz",
          "die rote Fingur plaustert",
          "und grausig gutzt der Golz.")

# text ist ein character vector; er sieht so aus:
text

# Die Funktion unnest_tokens() erlaubt die Transformation ins
# tidytext-Format. Zunächst wird der text in eine Tabelle ("tibble")
# umgewandelt:
text_df <- tibble(line = 1:4, text = text)
text_df

# Und dann mit unnest_tokens() tokenisiert (in einzelne Tokens aufgeteilt),
# in Kleinschreibung umgewandelt und die Interpunktion entfernt.
# Die Angabe der Zeilen, in der der Text stand, bleibt erhalten:
text_df %>%
  unnest_tokens(word, text)


####### Analyse von Wort- und Dokumenthäufigkeiten in einem Korpus ########
# Ein zentrales Ziel beim Text Mining ist es, durch Quantifizierung zu
# erforschen, um was es in einem Dokument geht. Zwei der hierbei häufig
# benutzten Maße sind tf und idf. 
#
# "term frequency": tf - Die Häufigkeit eines Wortes in einem Dokument,
# dividiert durch die absolute Anzahl dieses Wortes in diesem Dokument.
# Man spricht auch von relativer Vorkommenshäufigkeit.
# Aber: Es gibt Worte, die häufig in Dokumenten vorkommen, ohne besonders
# wichtig zu sein, wie z.B. im Englischen "the", "is", "of" usf. Diese Worte
# werden häufig auf einer Liste gesammelt ("stopwords") und vor der Analyse
# herausgenommen. Das macht häufig Sinn, aber es ist auch möglich, dass 
# einige dieser Worte in manchen Dokumenten wichtiger sind als in anderen.
#
# "inverse document frequency": idf - misst Bedeutung des Terms für
# die Gesamtmenge einer Dokumentsammlung; dabei wird die Gewichtung häufig
# benutzter Worte reduziert und die Gewichtung von Worten erhöht, die in einer 
# Dokumentensammlung nicht besonders häufig benutzt werden.
# tf-idf - Die beiden Mengen werden miteinander multipliziert; damit wird
# eine Gewichtung vorgenommen und die Häufigkeit eines Begriffs pro Dokument 
# mit der Seltenheit im Korpus abgeglichen. 
# Der statistische Wert tf-idf soll bemessen, wie wichtig ein Wort für ein
# Dokument bzw. in einem Korpus von Dokumenten ist, beispielsweise in einem
# Roman oder einer Sammlung von Romanen. 

# Das Paket tidytext wurde vorrangig für die Analyse englischsprachiger Texte 
# entwickelt, und alle Beispiele basieren darauf. Daher bietet das Paket
# auch nur die Entfernung von englischen Stopwords an.
# Aber: Textkorpora können mit den Paketen "tm" oder "quanteda" erstellt 
# und dann in das tidytext-Format importiert werden. Für die Exploration
# von Textkorpora in anderen gängigen Sprachen gibt
# es daher ebenfalls Lösungen (siehe "Resterampe").

# Das Arbeitsverzeichnis auswählen (d.h. den Basisordner für den Workshop)
# Session > Set Working Directory > Choose Directory

# Die Texte mit readtext einlesen
myCorpus <- readtext("CorpusKriegssammlung/*", encoding = "UTF-8")

# Das Korpus mit quanteda erstellen
ToxiCorpus <- corpus(myCorpus) 
class(ToxiCorpus)
# "corpus"    "character"

# Hinzufügen der Dokumentnamen
docvars(ToxiCorpus, 'book') <- docnames(ToxiCorpus)

# Ein Tokens Objekt erstellen
ToxiCorpus_token <- tokens(ToxiCorpus)

# Zahlen und Satzzeichen entfernen
ToxiCorpus_token <- tokens(ToxiCorpus_token, remove_numbers = TRUE, remove_punct = TRUE)

# nun setzen wir noch alles in Kleinschreibung (lowercasing)
ToxiCorpus_token <- tokens_tolower(ToxiCorpus_token)

# reinigen das Korpus noch ein wenig von OCR-Schrott
ToxiCorpus_token <-  tokens_remove(ToxiCorpus_token,
                                   pattern = c("^", "m", "s", "o", "b", "n", "h", "d", "f", "e", "w", "l", "k", 
                                               "g", "v", "u", "a", "t","z", "c", "r", "j", "i", "x",
                                               "co", "be", "de", "un", "en", "ge", "ver", "mk", "ii", "iii", "iv", "sic"),
                                   padding = TRUE) 
# und entfernen die Stopwords
ToxiCorpus_token <-  tokens_remove(ToxiCorpus_token,
                                   stopwords("german"),
                                   padding = TRUE) 

# Das Korpus mit quanteda in eine document-feature matrix konvertieren
dfmat_toxic <- dfm(ToxiCorpus_token)


# Immer noch vorhandene Leerzeichen aus der Matrix entfernen
dfmat_toxic_nostop <- dfm_select(dfmat_toxic, pattern = "",
                                 selection = "remove")

print(dfmat_toxic_nostop)

# Nun bringen wir die document-feature-matrix 
# mit der Funktion tidy() aus dem Paket Tidytext
# in das Tidy-Format
ToxiCorpusTidy <- tidy(dfmat_toxic_nostop) 

# Was ist das für ein Objekt?
class(ToxiCorpusTidy)
# Ein 'Tibble', bzw. ein data frame

# Einmal die Struktur anschauen
str(ToxiCorpusTidy)

# Und den Anfang der Tabelle in der tidy-Struktur anschauen
ToxiCorpusTidy

# Hier sortieren wir nach der Spalte "count", in absteigender Häufigkeit
ToxiCorpusTidy <- arrange(ToxiCorpusTidy, desc(count))

ToxiCorpusTidy

# Wir ermitteln die absolute Anzahl aller Wörter in einem Dokument
total_words <- ToxiCorpusTidy %>% 
  group_by(document) %>% 
  summarize(total = sum(count))

# Und fügen die Spalte zu unserem dataframe hinzu
ToxiCorpusTidy <- left_join(ToxiCorpusTidy, total_words)

ToxiCorpusTidy
# In diesem dataframe gibt es eine Zeile für jede document-term-Kombination;
# "count" gibt an, wie häufig dieser Begriff im Dokument vorkommt
# "total" gibt die Anzahl aller Wörter im Dokument an
# Die üblichen Verdächtigen sind natürlich "und", "der", "die" usf.,
# wenn die Stopwords nicht entfernt wurden.

# Wir schauen uns einmal die Verteilung count/total für jedes Dokument an,
# d.h. die Häufigkeit eines Begriffs geteilt durch die absolute Anzahl
# aller Wörter im Dokument - denn das ist genau, was "term frequency" beschreibt
# Das visualisieren wir
ggplot(ToxiCorpusTidy, aes(count/total, fill = document)) +
  geom_histogram(show.legend = FALSE) +
  xlim(NA, 0.0009) +
  facet_wrap(~document, ncol = 2, scales = "free_y")

# Hier findet sich immer dieselben Verteilungen wieder: 
# nur wenige Worte kommen in einem Text häufig vor, sie finden sich im Plot links
# Hingegen kommen viele Worte nur selten vor und finden sich in jedem Plot rechts
# das nennt man den "long tail"
# allerdings auch nicht wirklich hilfreiche Darstellung, da die Skalen
# auf der x-Achse unterschiedlich sind 

# tf-idf: Berechnung von tf-idf 
# Die Idee von tf-idf ist es, die wichtigsten Worte für den Inhalt eines
# jeden Dokuments zu finden, indem das Gewicht der häufigsten Worte verringert
# und das Gewicht seltener Worte in einem Korpus erhöht wird.
# Das schauen wir uns jetzt einmal an:
ToxiCorpusTidy_tf_idf <- ToxiCorpusTidy %>%
  bind_tf_idf(term, document, count)

ToxiCorpusTidy_tf_idf  
# Wie man sieht, sind idf (und daher auch tf_idf) gleich Null bei den 
# extrem häufigen Begriffen. Dies sind Worte, die in allen Dokumenten
# vorkommen, daher ist der idf-Wert (der 'Seltenheitswert') gleich null.

# Nun schauen wir uns Worte mit hoher tf-idf im Korpus an; 
# sie werden oben in der Tabelle angezeigt
ToxiCorpusTidy_tf_idf <- ToxiCorpusTidy %>%
  bind_tf_idf(term, document, count) %>%
  arrange(desc(tf_idf)) 

ToxiCorpusTidy_tf_idf
# Die 'seltenen' Worte sind erzählenden Texten in aller Regel Eigennamen, die in
# nur einem Text vorkommen. Keiner von ihnen kommt in allen Romanen vor.
# Daher sind das wichtige, charakteristische Worte für jeden der Texte
# im Korpus

# Das Ergebnis visualisieren wir einmal, sortiert nach tf_idf 
# Visualisiert werden die häufigsten Worte mit einem hohen tf-idf-Wert, 
# für jedes einzelne Dokument

# Die Auflösung 300 pixel per inch festlegen
ppi <- 300

tiff(paste0("tf-idf.tif"), width=24*ppi, height=48*ppi, compression = "lzw", res=ppi)
ToxiCorpusTidy_tf_idf %>%
  arrange(desc(tf_idf)) %>%
  mutate(term = factor(term, levels = rev(unique(term)))) %>% 
  group_by(document) %>% 
  top_n(5) %>% 
  ungroup %>%
  ggplot(aes(term, tf_idf, fill = document)) +
  geom_col(show.legend = FALSE) +
  labs(x = NULL, y = "tf-idf") +
  facet_wrap(~document, ncol = 2, scales = "free") +
  coord_flip()
dev.off()

# Sollte man Eigennamen entfernen?

# Eine ausführliche Einführung in das Paket Tidytext findet sich im Buch
# "Text Mining with R: A Tidy Appproach"
# online verfügbar unter https://www.tidytextmining.com/index.html
